// [License]
// The Ariba-Underlay Copyright
//
// Copyright (c) 2008-2009, Institute of Telematics, Universität Karlsruhe (TH)
//
// Institute of Telematics
// Universität Karlsruhe (TH)
// Zirkel 2, 76128 Karlsruhe
// Germany
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE INSTITUTE OF TELEMATICS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE INSTITUTE OF TELEMATICS OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// The views and conclusions contained in the software and documentation
// are those of the authors and should not be interpreted as representing
// official policies, either expressed or implied, of the Institute of
// Telematics.
// [License]

#ifndef SPOVNETPROPERTIES_H_
#define SPOVNETPROPERTIES_H_

#include <string>
#include <iostream>
#include <streambuf>

using std::string;

namespace ariba {
// forward declaration
class SpoVNetProperties;
}

#include "Identifiers.h"
#include "Name.h"

namespace ariba {

/**
 * This class implements a container that holds all properties of a
 * SpoVNet instance. It may evolve with new features when new features
 * are introduced.
 *
 * @author Sebastian Mies <mies@tm.uka.de>
 * @author Christoph Mayer <mayer@tm.uka.de>
 */
class SpoVNetProperties {
public:
	enum OverlayType {
		ONE_HOP_OVERLAY = 0,
		CHORD_OVERLAY = 1,
	};


	/**
	 * This object holds the default settings for a newly created spovnet
	 * instance.
	 */
	static const SpoVNetProperties DEFAULT;

	/**
	 * Constructs a new default SpoVnet property object.
	 */
	SpoVNetProperties();

	/**
	 * Constructs a new SpoVnet property object.
	 *
	 * TODO: replace with setters! for downwards compatibility
	 */
	/*
	SpoVNetProperties(const Name& name, SpoVNetID id, OverlayType type,
			uint16_t idLength, const NodeID& initiator, bool hidden = false) :
		name(name), id(id), type(type), idLength(idLength),
				initiator(initiator), hidden(hidden) {
	}*/

	/**
	 * Copy constructor.
	 */
	SpoVNetProperties(const SpoVNetProperties& copy);

	/**
	 * Destructor.
	 */
	virtual ~SpoVNetProperties();

	/**
	 * Returns the canonical SpoVNet name
	 */
	const Name& getName() const;

	/**
	 * Returns the SpoVNet id
	 */
	const SpoVNetID& getId() const;

	/**
	 * Returns the node id of the initiator of the spovnet.
	 * If the node id is unspecified, the initiator wanted to be anonymous.
	 */
	const NodeID& getInitiator() const;

	/**
	 * Returns the node identifier length in bites
	 */
	uint16_t getIdentifierLength() const;

	void setIdentifierLength( uint16_t length ) {
		this->idLength = length;
	}

	/**
	 * Returns the overlay type.
	 */
	const OverlayType getBaseOverlayType() const;

	void setBaseOverlayType( OverlayType type ) {
		this->type = type;
	}

	/**
	 * Returns true, if the spovnet is hidden
	 */
	bool isHidden() const;

	void setHidden( bool is_hidden ) {
		this->hidden = is_hidden;
	}

	/**
	 * Returns a human readable string representation of the SpoVNet properties
	 *
	 * @return A human readable string representation of the SpoVNet properties
	 */
	std::string toString() const;

private:
	Name name;
	SpoVNetID id;
	uint8_t  type;
	uint16_t idLength;
	NodeID initiator;
	bool hidden;
};

} // namespace ariba

#endif /* SPOVNETPROPERTIES_H_ */
