// [License]
// The Ariba-Underlay Copyright
//
// Copyright (c) 2008-2009, Institute of Telematics, Universität Karlsruhe (TH)
//
// Institute of Telematics
// Universität Karlsruhe (TH)
// Zirkel 2, 76128 Karlsruhe
// Germany
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE INSTITUTE OF TELEMATICS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE ARIBA PROJECT OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// The views and conclusions contained in the software and documentation
// are those of the authors and should not be interpreted as representing
// official policies, either expressed or implied, of the Institute of
// Telematics.
// [License]

#ifndef IPV4LOCATOR_H_
#define IPV4LOCATOR_H_

#include <string>

#include "ariba/communication/modules/network/NetworkLocator.h"
#include <boost/asio/ip/address_v4.hpp>

using std::string;
using ariba::communication::NetworkLocator;

namespace ariba {
namespace communication {

/**
 * An address class for IPv4 locators (= addresses).
 *
 * This class handles IPv4 locators only.
 *
 */
class IPv4Locator: public NetworkLocator {
	VSERIALIZEABLE;
public:
	/** Constructor initializing the address to the one given in string representation. */
	IPv4Locator();
	IPv4Locator(const IPv4Locator& rh);

	static IPv4Locator LOCALHOST;
	static IPv4Locator ANY;
	static IPv4Locator BROADCAST;

	virtual bool operator==(const NetworkLocator& rh) const;
	virtual bool operator!=(const NetworkLocator& rh) const;

	/** Default destructor. */
	virtual ~IPv4Locator();

	static IPv4Locator fromString(string addr);
	virtual string toString() const;

	void setIP( string ip ) {
		ipv4Address = boost::asio::ip::address_v4::from_string( ip );
	}

	string getIP() const {
		return ipv4Address.to_string();
	}

	void setPort( uint16_t port ) {
		this->port = port;
	}

	uint16_t getPort() const {
		return this->port;
	}
private:
	boost::asio::ip::address_v4 ipv4Address;
	uint16_t port;
};

}} // namespace ariba, communication

sznBeginDefault( ariba::communication::IPv4Locator, X ) {
	boost::asio::ip::address_v4::bytes_type buffer = ipv4Address.to_bytes();
	for (int i=0; i<4; i++) X && buffer[i];
	if (X.isDeserializer()) ipv4Address = boost::asio::ip::address_v4(buffer);
	X && port;
} sznEnd();

#endif /* IPV4LOCATOR_H_ */
