/*
 * tcpip_endpoint.cpp
 *
 *  Created on: 26.03.2013
 *      Author: mario
 */

#include "tcpip_endpoint.hpp"

// system
#include <sstream>
#include <stdexcept>

using namespace std;
using namespace boost::asio::ip;

namespace ariba {
namespace addressing2 {

#define TCPIPv6_SIZE 18
#define TCPIPv4_SIZE 6


/// factories
TcpIP_EndpointPtr tcpip_endpoint::create_TcpIP_Endpoint(
        const tcp::endpoint& asio_endpoint)
{
    TcpIP_EndpointPtr ptr(new tcpip_endpoint(asio_endpoint));
    
    return ptr;
}


/// constructors
tcpip_endpoint::tcpip_endpoint(const std::string& ip_addr, const int port)
{
    address addr = address::from_string(ip_addr);
    asio_endpoint = tcp::endpoint(addr, static_cast<uint16_t>(port));
}

tcpip_endpoint::tcpip_endpoint(const ENDPOINT_TYPE type, const uint8_t* const byte_array, const int read_max)
{
    // IPv4
    if ( type == endpoint_type::TCPIPv4 )
    {
        // boundary check
        if ( read_max < TCPIPv4_SIZE )
            throw invalid_argument("Not enough bytes to read an TCPIPv4 endpoint.");
        
        asio_endpoint = tcp::endpoint( address_v4( *((uint32_t*) byte_array) ),
                                        *((uint16_t*) (byte_array+sizeof(uint32_t))) );
    }
    
    // IPv6
    else if ( type == endpoint_type::TCPIPv6 )
    {
        // boundary check
        if ( read_max < TCPIPv6_SIZE )
            throw invalid_argument("Not enough bytes to read an TCPIPv6 endpoint.");

        address_v6::bytes_type bytes_;
        for (size_t i=0; i<bytes_.size(); i++)
        {
            bytes_[i] = byte_array[i];
        }
        
        asio_endpoint = tcp::endpoint( address_v6(bytes_),
                                        *((uint16_t*) (byte_array+bytes_.size())) );
    }
    
    // Error
    else
    {
        throw invalid_argument("Corrupt data. Can't deserialize endpoint.");
    }
}

tcpip_endpoint::tcpip_endpoint(const tcp::endpoint& asio_endpoint)  :
        asio_endpoint(asio_endpoint)
{
}


tcpip_endpoint::~tcpip_endpoint()
{
}



ENDPOINT_CATEGORY tcpip_endpoint::get_category() const
{
    return endpoint_category::TCPIP;
}

ENDPOINT_TYPE tcpip_endpoint::get_type() const
{
    // IPv4
    if ( asio_endpoint.address().is_v4() )
    {
        return endpoint_type::TCPIPv4;
    }
    
    // IPv6
    else if ( asio_endpoint.address().is_v6() )
    {
        return endpoint_type::TCPIPv6;
    }

    // Invalid
    else
    {
        return endpoint_type::INVALID;
    }
}


std::string tcpip_endpoint::to_string() const
{
    ostringstream out;
    
    out << asio_endpoint;
    
    return out.str();
}



size_t tcpip_endpoint::to_byte_array(uint8_t* buffer) const
{
    // IPv4
    if ( asio_endpoint.address().is_v4() )
    {
        uint32_t ip = asio_endpoint.address().to_v4().to_ulong();
        uint16_t port = asio_endpoint.port();
        
        memcpy(buffer, &ip, sizeof(uint32_t));
        memcpy(buffer+sizeof(uint32_t), &port, sizeof(uint16_t));
        
        return TCPIPv4_SIZE;
    }
    
    // IPv6
    else
    {
        address_v6::bytes_type bytes_ = asio_endpoint.address().to_v6().to_bytes();
        
        for (size_t i=0; i<bytes_.size(); i++)
        {
            buffer[i] = bytes_[i];
        }
        
        uint16_t port = asio_endpoint.port();
        memcpy(buffer+bytes_.size(), &port, sizeof(uint16_t));
        
        return TCPIPv6_SIZE;
    }

}


int tcpip_endpoint::size() const
{
    // IPv4
    if ( asio_endpoint.protocol() == tcp::v4() )
        return TCPIPv4_SIZE;
    
    // IPv6
    else
        return TCPIPv6_SIZE;
}

const tcp::endpoint& tcpip_endpoint::to_asio() const
{
    return asio_endpoint;
}

bool tcpip_endpoint::equals(const TcpIP_EndpointPtr& rhs) const
{
    return asio_endpoint == rhs->asio_endpoint;
}

}} /* namespace ariba::addressing2 */
